// Numus Logistics - Website Templates
// Generates HTML for customer-facing logistics websites

function generateWebsiteHTML(config) {
  const {
    template,
    logo,
    businessName,
    tagline,
    colors,
    sections,
    contact,
    social,
    subdomain
  } = config;

  const templates = {
    modern: getModernTemplate,
    minimal: getMinimalTemplate,
    bold: getBoldTemplate,
    corporate: getCorporateTemplate
  };

  const getTemplate = templates[template] || templates.modern;
  return getTemplate(config);
}

function getModernTemplate(config) {
  const { logo, businessName, tagline, colors, sections, contact, social, subdomain, hero, services, pricing, testimonials } = config;
  
  return `
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>${businessName || 'Logistics Company'} - Fast & Reliable Delivery</title>
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap" rel="stylesheet">
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    body {
      font-family: 'Inter', sans-serif;
      color: ${colors.text};
      background: ${colors.background};
      line-height: 1.6;
    }
    .container { max-width: 1200px; margin: 0 auto; padding: 0 20px; }
    
    /* Header */
    header {
      background: ${colors.background};
      padding: 16px 0;
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      z-index: 100;
      box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }
    .header-inner {
      display: flex;
      justify-content: space-between;
      align-items: center;
    }
    .logo {
      display: flex;
      align-items: center;
      gap: 10px;
      font-size: 24px;
      font-weight: 800;
      color: ${colors.primary};
      text-decoration: none;
    }
    .logo img { height: 40px; }
    nav { display: flex; gap: 32px; align-items: center; }
    nav a {
      color: ${colors.text};
      text-decoration: none;
      font-weight: 500;
      font-size: 15px;
      transition: color 0.2s;
    }
    nav a:hover { color: ${colors.primary}; }
    .btn {
      display: inline-flex;
      align-items: center;
      gap: 8px;
      padding: 12px 24px;
      border-radius: 8px;
      font-weight: 600;
      text-decoration: none;
      transition: all 0.2s;
      border: none;
      cursor: pointer;
      font-size: 15px;
    }
    .btn-primary {
      background: ${colors.primary};
      color: white;
    }
    .btn-primary:hover { opacity: 0.9; transform: translateY(-1px); }
    .btn-accent {
      background: ${colors.accent};
      color: white;
    }
    .btn-outline {
      background: transparent;
      border: 2px solid ${colors.primary};
      color: ${colors.primary};
    }
    
    /* Hero */
    .hero {
      padding: 160px 0 100px;
      background: linear-gradient(135deg, ${colors.primary}08 0%, ${colors.accent}08 100%);
    }
    .hero-content {
      display: grid;
      grid-template-columns: 1fr 1fr;
      gap: 60px;
      align-items: center;
    }
    .hero h1 {
      font-size: 56px;
      font-weight: 800;
      line-height: 1.1;
      margin-bottom: 24px;
      color: ${colors.text};
    }
    .hero h1 span { color: ${colors.accent}; }
    .hero p {
      font-size: 18px;
      color: ${colors.text}99;
      margin-bottom: 32px;
    }
    .hero-buttons { display: flex; gap: 16px; }
    .hero-image {
      display: flex;
      justify-content: center;
    }
    .hero-image-placeholder {
      width: 100%;
      max-width: 500px;
      aspect-ratio: 1;
      background: linear-gradient(135deg, ${colors.primary} 0%, ${colors.accent} 100%);
      border-radius: 24px;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 120px;
    }
    
    /* Services */
    .services {
      padding: 100px 0;
    }
    .section-header {
      text-align: center;
      margin-bottom: 60px;
    }
    .section-header h2 {
      font-size: 40px;
      font-weight: 700;
      margin-bottom: 16px;
    }
    .section-header p {
      font-size: 18px;
      color: ${colors.text}80;
      max-width: 600px;
      margin: 0 auto;
    }
    .services-grid {
      display: grid;
      grid-template-columns: repeat(3, 1fr);
      gap: 32px;
    }
    .service-card {
      background: white;
      padding: 32px;
      border-radius: 16px;
      box-shadow: 0 4px 20px rgba(0,0,0,0.05);
      transition: transform 0.2s, box-shadow 0.2s;
    }
    .service-card:hover {
      transform: translateY(-4px);
      box-shadow: 0 8px 30px rgba(0,0,0,0.1);
    }
    .service-icon {
      width: 64px;
      height: 64px;
      background: ${colors.primary}15;
      border-radius: 16px;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 28px;
      margin-bottom: 20px;
    }
    .service-card h3 {
      font-size: 20px;
      margin-bottom: 12px;
    }
    .service-card p {
      color: ${colors.text}80;
      font-size: 15px;
    }
    
    /* Tracking */
    .tracking {
      padding: 100px 0;
      background: ${colors.primary};
      color: white;
    }
    .tracking h2 {
      font-size: 36px;
      font-weight: 700;
      margin-bottom: 16px;
      text-align: center;
    }
    .tracking > p {
      text-align: center;
      opacity: 0.8;
      margin-bottom: 40px;
    }
    .tracking-form {
      max-width: 600px;
      margin: 0 auto;
      display: flex;
      gap: 12px;
    }
    .tracking-form input {
      flex: 1;
      padding: 16px 20px;
      border: none;
      border-radius: 8px;
      font-size: 16px;
    }
    .tracking-form button {
      padding: 16px 32px;
      background: ${colors.accent};
      color: white;
      border: none;
      border-radius: 8px;
      font-weight: 600;
      cursor: pointer;
      font-size: 16px;
    }
    
    /* Pricing */
    .pricing {
      padding: 100px 0;
    }
    .pricing-grid {
      display: grid;
      grid-template-columns: repeat(3, 1fr);
      gap: 32px;
      max-width: 1000px;
      margin: 0 auto;
    }
    .pricing-card {
      background: white;
      border: 2px solid ${colors.primary}20;
      border-radius: 16px;
      padding: 40px;
      text-align: center;
    }
    .pricing-card.featured {
      border-color: ${colors.primary};
      position: relative;
    }
    .pricing-card.featured::before {
      content: 'Popular';
      position: absolute;
      top: -12px;
      left: 50%;
      transform: translateX(-50%);
      background: ${colors.primary};
      color: white;
      padding: 4px 16px;
      border-radius: 20px;
      font-size: 13px;
      font-weight: 600;
    }
    .pricing-card h3 {
      font-size: 24px;
      margin-bottom: 8px;
    }
    .pricing-card .price {
      font-size: 48px;
      font-weight: 800;
      color: ${colors.primary};
      margin: 20px 0;
    }
    .pricing-card .price span {
      font-size: 16px;
      font-weight: 400;
      color: ${colors.text}80;
    }
    .pricing-card ul {
      list-style: none;
      text-align: left;
      margin: 24px 0;
    }
    .pricing-card li {
      padding: 8px 0;
      display: flex;
      align-items: center;
      gap: 8px;
    }
    .pricing-card li::before {
      content: '✓';
      color: ${colors.accent};
      font-weight: bold;
    }

    /* Testimonials */
    .testimonials {
      padding: 100px 0;
      background: ${colors.primary}05;
    }
    .testimonials-grid {
      display: grid;
      grid-template-columns: repeat(3, 1fr);
      gap: 32px;
      margin-top: 60px;
    }
    .testimonial-card {
      background: white;
      padding: 32px;
      border-radius: 16px;
      box-shadow: 0 4px 20px rgba(0,0,0,0.05);
    }
    .testimonial-rating {
      color: ${colors.accent};
      font-size: 20px;
      margin-bottom: 16px;
    }
    .testimonial-text {
      font-size: 15px;
      line-height: 1.7;
      color: ${colors.text}CC;
      margin-bottom: 24px;
      font-style: italic;
    }
    .testimonial-author {
      display: flex;
      align-items: center;
      gap: 12px;
    }
    .testimonial-avatar {
      width: 48px;
      height: 48px;
      border-radius: 50%;
      background: ${colors.primary}20;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 20px;
    }
    .testimonial-author-name {
      font-weight: 600;
      margin-bottom: 2px;
    }
    .testimonial-author-role {
      font-size: 13px;
      color: ${colors.text}80;
    }

    /* Contact */
    .contact {
      padding: 100px 0;
      background: ${colors.primary}05;
    }
    .contact-grid {
      display: grid;
      grid-template-columns: 1fr 1fr;
      gap: 60px;
    }
    .contact-info h3 {
      font-size: 32px;
      margin-bottom: 24px;
    }
    .contact-item {
      display: flex;
      align-items: flex-start;
      gap: 16px;
      margin-bottom: 24px;
    }
    .contact-item-icon {
      width: 48px;
      height: 48px;
      background: ${colors.primary};
      border-radius: 12px;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 20px;
    }
    .contact-item h4 {
      font-size: 14px;
      color: ${colors.text}80;
      margin-bottom: 4px;
    }
    .contact-item p {
      font-size: 16px;
      font-weight: 500;
    }
    .contact-form {
      background: white;
      padding: 40px;
      border-radius: 16px;
      box-shadow: 0 4px 20px rgba(0,0,0,0.05);
    }
    .form-group {
      margin-bottom: 20px;
    }
    .form-group label {
      display: block;
      font-weight: 500;
      margin-bottom: 8px;
      font-size: 14px;
    }
    .form-group input, .form-group textarea {
      width: 100%;
      padding: 14px 16px;
      border: 1px solid ${colors.text}20;
      border-radius: 8px;
      font-size: 15px;
      font-family: inherit;
    }
    .form-group textarea { resize: vertical; min-height: 120px; }
    
    /* Footer */
    footer {
      background: ${colors.text};
      color: white;
      padding: 60px 0 30px;
    }
    .footer-grid {
      display: grid;
      grid-template-columns: 2fr 1fr 1fr 1fr;
      gap: 40px;
      margin-bottom: 40px;
    }
    .footer-brand p {
      opacity: 0.7;
      margin: 16px 0;
      font-size: 14px;
    }
    .footer-social {
      display: flex;
      gap: 12px;
    }
    .footer-social a {
      width: 40px;
      height: 40px;
      background: rgba(255,255,255,0.1);
      border-radius: 8px;
      display: flex;
      align-items: center;
      justify-content: center;
      text-decoration: none;
      font-size: 18px;
    }
    footer h4 {
      font-size: 16px;
      margin-bottom: 20px;
    }
    footer ul { list-style: none; }
    footer li { margin-bottom: 12px; }
    footer a {
      color: rgba(255,255,255,0.7);
      text-decoration: none;
      font-size: 14px;
    }
    footer a:hover { color: white; }
    .footer-bottom {
      border-top: 1px solid rgba(255,255,255,0.1);
      padding-top: 30px;
      text-align: center;
      opacity: 0.6;
      font-size: 14px;
    }
    
    /* Responsive */
    @media (max-width: 768px) {
      nav { display: none; }
      .hero-content { grid-template-columns: 1fr; text-align: center; }
      .hero h1 { font-size: 36px; }
      .hero-buttons { justify-content: center; }
      .services-grid { grid-template-columns: 1fr; }
      .testimonials-grid { grid-template-columns: 1fr; }
      .pricing-grid { grid-template-columns: 1fr; }
      .contact-grid { grid-template-columns: 1fr; }
      .footer-grid { grid-template-columns: 1fr 1fr; }
      .tracking-form { flex-direction: column; }
    }
  </style>
</head>
<body>
  <header>
    <div class="container header-inner">
      <a href="#" class="logo">
        ${logo ? `<img src="${logo}" alt="${businessName}">` : '📦'}
        ${businessName || 'LogiCo'}
      </a>
      <nav>
        ${sections.services ? '<a href="#services">Services</a>' : ''}
        ${sections.tracking ? '<a href="#tracking">Track</a>' : ''}
        ${sections.pricing ? '<a href="#pricing">Pricing</a>' : ''}
        ${sections.testimonials ? '<a href="#testimonials">Reviews</a>' : ''}
        ${sections.contact ? '<a href="#contact">Contact</a>' : ''}
        <a href="#book" class="btn btn-primary">Book Delivery</a>
      </nav>
    </div>
  </header>

  ${sections.hero ? `
  <section class="hero">
    <div class="container">
      <div class="hero-content">
        <div>
          <h1>${hero?.title || tagline || 'Fast & <span>Reliable</span> Delivery'}</h1>
          <p>${hero?.description || 'We deliver your packages safely and on time. Track your shipments in real-time and enjoy seamless logistics services.'}</p>
          <div class="hero-buttons">
            <a href="#book" class="btn btn-primary">📦 Book a Delivery</a>
            <a href="#tracking" class="btn btn-outline">🔍 Track Order</a>
          </div>
        </div>
        <div class="hero-image">
          <div class="hero-image-placeholder">🚚</div>
        </div>
      </div>
    </div>
  </section>
  ` : ''}

  ${sections.services ? `
  <section class="services" id="services">
    <div class="container">
      <div class="section-header">
        <h2>Our Services</h2>
        <p>We offer comprehensive logistics solutions tailored to your needs</p>
      </div>
      <div class="services-grid">
        <div class="service-card">
          <div class="service-icon">📦</div>
          <h3>${services?.[0]?.title || 'Express Delivery'}</h3>
          <p>${services?.[0]?.description || 'Same-day and next-day delivery options for urgent shipments within the city.'}</p>
        </div>
        <div class="service-card">
          <div class="service-icon">🚛</div>
          <h3>${services?.[1]?.title || 'Interstate Shipping'}</h3>
          <p>${services?.[1]?.description || 'Reliable delivery services across states with tracking and insurance options.'}</p>
        </div>
        <div class="service-card">
          <div class="service-icon">🏢</div>
          <h3>${services?.[2]?.title || 'Business Logistics'}</h3>
          <p>${services?.[2]?.description || 'Dedicated logistics solutions for e-commerce and business clients.'}</p>
        </div>
      </div>
    </div>
  </section>
  ` : ''}

  ${sections.tracking ? `
  <section class="tracking" id="tracking">
    <div class="container">
      <h2>Track Your Order</h2>
      <p>Enter your tracking number to see real-time updates</p>
      <form class="tracking-form" onsubmit="trackOrder(event)">
        <input type="text" placeholder="Enter tracking number (e.g., NL-2024-0001)" id="trackingInput">
        <button type="submit">Track</button>
      </form>
      <div id="trackingResult" style="max-width: 600px; margin: 30px auto 0; text-align: center;"></div>
    </div>
  </section>
  ` : ''}

  ${sections.pricing ? `
  <section class="pricing" id="pricing">
    <div class="container">
      <div class="section-header">
        <h2>Simple Pricing</h2>
        <p>Transparent pricing with no hidden fees</p>
      </div>
      <div class="pricing-grid">
        <div class="pricing-card">
          <h3>${pricing?.[0]?.name || 'Local'}</h3>
          <div class="price">${pricing?.[0]?.price || '₦1,500'}<span>/delivery</span></div>
          <ul>
            <li>Within city limits</li>
            <li>Same-day delivery</li>
            <li>Up to 5kg</li>
            <li>Real-time tracking</li>
          </ul>
          <a href="#book" class="btn btn-outline" style="width: 100%;">Get Started</a>
        </div>
        <div class="pricing-card featured">
          <h3>${pricing?.[1]?.name || 'Express'}</h3>
          <div class="price">${pricing?.[1]?.price || '₦3,000'}<span>/delivery</span></div>
          <ul>
            <li>Priority handling</li>
            <li>2-hour delivery</li>
            <li>Up to 10kg</li>
            <li>SMS notifications</li>
          </ul>
          <a href="#book" class="btn btn-primary" style="width: 100%;">Get Started</a>
        </div>
        <div class="pricing-card">
          <h3>${pricing?.[2]?.name || 'Interstate'}</h3>
          <div class="price">${pricing?.[2]?.price || '₦5,000+'}<span>/delivery</span></div>
          <ul>
            <li>Nationwide coverage</li>
            <li>2-5 business days</li>
            <li>Up to 30kg</li>
            <li>Insurance included</li>
          </ul>
          <a href="#book" class="btn btn-outline" style="width: 100%;">Get Started</a>
        </div>
      </div>
    </div>
  </section>
  ` : ''}

  ${sections.testimonials ? `
  <section class="testimonials" id="testimonials">
    <div class="container">
      <div class="section-header">
        <h2>What Our Clients Say</h2>
        <p>Trusted by businesses and individuals across the nation</p>
      </div>
      <div class="testimonials-grid">
        <div class="testimonial-card">
          <div class="testimonial-rating">⭐⭐⭐⭐⭐</div>
          <div class="testimonial-text">
            "${testimonials?.[0]?.text || 'Excellent service! My packages always arrive on time and in perfect condition. The real-time tracking gives me peace of mind.'}"
          </div>
          <div class="testimonial-author">
            <div class="testimonial-avatar">👤</div>
            <div>
              <div class="testimonial-author-name">${testimonials?.[0]?.name || 'Sarah Johnson'}</div>
              <div class="testimonial-author-role">${testimonials?.[0]?.role || 'E-commerce Owner'}</div>
            </div>
          </div>
        </div>
        <div class="testimonial-card">
          <div class="testimonial-rating">⭐⭐⭐⭐⭐</div>
          <div class="testimonial-text">
            "${testimonials?.[1]?.text || 'Professional and reliable logistics partner. They\'ve helped scale our business with their efficient delivery network.'}"
          </div>
          <div class="testimonial-author">
            <div class="testimonial-avatar">👤</div>
            <div>
              <div class="testimonial-author-name">${testimonials?.[1]?.name || 'Michael Chen'}</div>
              <div class="testimonial-author-role">${testimonials?.[1]?.role || 'Business Director'}</div>
            </div>
          </div>
        </div>
        <div class="testimonial-card">
          <div class="testimonial-rating">⭐⭐⭐⭐⭐</div>
          <div class="testimonial-text">
            "${testimonials?.[2]?.text || 'Fast, affordable, and customer-focused. I\'ve been using their services for over a year and never disappointed!'}"
          </div>
          <div class="testimonial-author">
            <div class="testimonial-avatar">👤</div>
            <div>
              <div class="testimonial-author-name">${testimonials?.[2]?.name || 'Aisha Mohammed'}</div>
              <div class="testimonial-author-role">${testimonials?.[2]?.role || 'Frequent Customer'}</div>
            </div>
          </div>
        </div>
      </div>
    </div>
  </section>
  ` : ''}

  ${sections.contact ? `
  <section class="contact" id="contact">
    <div class="container">
      <div class="contact-grid">
        <div class="contact-info">
          <h3>Get in Touch</h3>
          <p style="color: ${colors.text}80; margin-bottom: 32px;">Have questions? We're here to help 24/7</p>
          
          ${contact.phone ? `
          <div class="contact-item">
            <div class="contact-item-icon">📞</div>
            <div>
              <h4>Phone</h4>
              <p>${contact.phone}</p>
            </div>
          </div>
          ` : ''}
          
          ${contact.email ? `
          <div class="contact-item">
            <div class="contact-item-icon">✉️</div>
            <div>
              <h4>Email</h4>
              <p>${contact.email}</p>
            </div>
          </div>
          ` : ''}
          
          ${contact.whatsapp ? `
          <div class="contact-item">
            <div class="contact-item-icon">💬</div>
            <div>
              <h4>WhatsApp</h4>
              <p>${contact.whatsapp}</p>
            </div>
          </div>
          ` : ''}
          
          ${contact.address ? `
          <div class="contact-item">
            <div class="contact-item-icon">📍</div>
            <div>
              <h4>Address</h4>
              <p>${contact.address}</p>
            </div>
          </div>
          ` : ''}
        </div>
        
        <form class="contact-form" onsubmit="submitContact(event)">
          <div class="form-group">
            <label>Your Name</label>
            <input type="text" placeholder="John Doe" required>
          </div>
          <div class="form-group">
            <label>Phone Number</label>
            <input type="tel" placeholder="+234 800 000 0000" required>
          </div>
          <div class="form-group">
            <label>Message</label>
            <textarea placeholder="How can we help you?"></textarea>
          </div>
          <button type="submit" class="btn btn-primary" style="width: 100%;">Send Message</button>
        </form>
      </div>
    </div>
  </section>
  ` : ''}

  <footer>
    <div class="container">
      <div class="footer-grid">
        <div class="footer-brand">
          <div class="logo" style="color: white;">
            ${logo ? `<img src="${logo}" alt="${businessName}" style="height: 40px;">` : '📦'}
            ${businessName || 'LogiCo'}
          </div>
          <p>${tagline || 'Fast & reliable delivery services'}</p>
          <div class="footer-social">
            ${social.instagram ? `<a href="https://instagram.com/${social.instagram.replace('@', '')}" target="_blank">📷</a>` : ''}
            ${social.twitter ? `<a href="https://twitter.com/${social.twitter.replace('@', '')}" target="_blank">🐦</a>` : ''}
            ${social.facebook ? `<a href="${social.facebook.includes('http') ? social.facebook : 'https://facebook.com/' + social.facebook}" target="_blank">👤</a>` : ''}
          </div>
        </div>
        <div>
          <h4>Services</h4>
          <ul>
            <li><a href="#">Local Delivery</a></li>
            <li><a href="#">Express Delivery</a></li>
            <li><a href="#">Interstate</a></li>
            <li><a href="#">Business</a></li>
          </ul>
        </div>
        <div>
          <h4>Company</h4>
          <ul>
            <li><a href="#">About Us</a></li>
            <li><a href="#">Careers</a></li>
            <li><a href="#">Terms of Service</a></li>
            <li><a href="#">Privacy Policy</a></li>
          </ul>
        </div>
        <div>
          <h4>Support</h4>
          <ul>
            <li><a href="#contact">Contact Us</a></li>
            <li><a href="#tracking">Track Order</a></li>
            <li><a href="#">FAQs</a></li>
            <li><a href="#">Help Center</a></li>
          </ul>
        </div>
      </div>
      <div class="footer-bottom">
        <p>© ${new Date().getFullYear()} ${businessName || 'LogiCo'}. All rights reserved. Powered by Numus</p>
      </div>
    </div>
  </footer>

  <script>
    function trackOrder(e) {
      e.preventDefault();
      const trackingId = document.getElementById('trackingInput').value;
      const resultDiv = document.getElementById('trackingResult');
      resultDiv.innerHTML = '<p style="color: white;">🔍 Searching for order ' + trackingId + '...</p>';
      
      // This would call your actual API
      setTimeout(() => {
        resultDiv.innerHTML = '<p style="color: white;">📦 Order found! Status: <strong>In Transit</strong> - Estimated delivery: Today by 5:00 PM</p>';
      }, 1500);
    }
    
    function submitContact(e) {
      e.preventDefault();
      alert('Message sent successfully! We will get back to you soon.');
      e.target.reset();
    }
  </script>
</body>
</html>
  `;
}

function getMinimalTemplate(config) {
  // Minimal clean template
  const { logo, businessName, tagline, colors, sections, contact, social } = config;
  // Similar structure but with minimal styling
  return getModernTemplate(config); // Fallback for now
}

function getBoldTemplate(config) {
  // Bold colorful template
  return getModernTemplate(config);
}

function getCorporateTemplate(config) {
  // Professional corporate template
  return getModernTemplate(config);
}
